<?php
require_once __DIR__ . '/../core/auth.php';
require_login();
$pdo = db();

/* ================= CSRF ================= */
if (empty($_SESSION['csrf'])) {
  $_SESSION['csrf'] = bin2hex(random_bytes(16));
}
$CSRF = $_SESSION['csrf'];

/* ================= AJAX API =================
   - ?action=get&branch_id=ID           => جلب إعدادات فرع
   - POST action=save (JSON)            => حفظ إعدادات فرع
================================================ */
function json_out($data, $code=200){
  http_response_code($code);
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode($data, JSON_UNESCAPED_UNICODE);
  exit;
}

if (isset($_GET['action']) && $_GET['action']==='get') {
  $bid = (int)($_GET['branch_id'] ?? 0);
  // اسم الفرع للمعاينة
  $stmt = $pdo->prepare("SELECT BranchName FROM branches WHERE BranchID=? AND IsActive=1");
  $stmt->execute([$bid]);
  $branchRow = $stmt->fetch(PDO::FETCH_ASSOC);
  if (!$branchRow) json_out(['ok'=>false,'msg'=>'فرع غير موجود أو غير نشط'], 404);

  $st = $pdo->prepare("SELECT BranchID, TaxRate, ServicePercent, Currency, PrintFooter
                       FROM BranchSettings WHERE BranchID=?");
  $st->execute([$bid]);
  $bs = $st->fetch(PDO::FETCH_ASSOC);

  if (!$bs) {
    // قيم افتراضية في حالة عدم وجود سجل بعد
    $bs = [
      'BranchID'       => $bid,
      'TaxRate'        => 0.00,
      'ServicePercent' => 0.00,
      'Currency'       => 'EGP',
      'PrintFooter'    => null
    ];
  }
  $bs['BranchName'] = $branchRow['BranchName'];
  json_out(['ok'=>true,'data'=>$bs]);
}

if (isset($_POST['action']) && $_POST['action']==='save') {
  // الطلب جاي كـ JSON
  $payload = json_decode(file_get_contents('php://input'), true) ?: [];
  if (($payload['csrf'] ?? '') !== ($_SESSION['csrf'] ?? '')) {
    json_out(['ok'=>false,'msg'=>'CSRF غير صالح'], 403);
  }

  $BranchID       = (int)($payload['BranchID'] ?? 0);
  $TaxRate        = isset($payload['TaxRate']) ? (float)$payload['TaxRate'] : 0.0;
  $ServicePercent = isset($payload['ServicePercent']) ? (float)$payload['ServicePercent'] : 0.0;
  $Currency       = trim((string)($payload['Currency'] ?? 'EGP'));
  $PrintFooter    = trim((string)($payload['PrintFooter'] ?? ''));

  // قيود بسيطة
  if ($BranchID <= 0) json_out(['ok'=>false,'msg'=>'فرع غير محدد'], 422);
  if ($TaxRate < 0) $TaxRate = 0;
  if ($TaxRate > 100) $TaxRate = 100;
  if ($ServicePercent < 0) $ServicePercent = 0;
  if ($ServicePercent > 100) $ServicePercent = 100;
  if ($Currency === '') $Currency = 'EGP';
  if ($PrintFooter === '') $PrintFooter = null;

  // تأكد الفرع موجود ونشط
  $chk = $pdo->prepare("SELECT 1 FROM branches WHERE BranchID=? AND IsActive=1");
  $chk->execute([$BranchID]);
  if (!$chk->fetchColumn()) json_out(['ok'=>false,'msg'=>'الفرع غير موجود أو غير نشط'], 404);

  // UPSERT: لو موجود نحدّث، غير كده ننشئ
  $exists = $pdo->prepare("SELECT 1 FROM BranchSettings WHERE BranchID=?");
  $exists->execute([$BranchID]);
  if ($exists->fetchColumn()) {
    $q = "UPDATE BranchSettings
          SET TaxRate=?, ServicePercent=?, Currency=?, PrintFooter=?
          WHERE BranchID=?";
    $pdo->prepare($q)->execute([$TaxRate, $ServicePercent, $Currency, $PrintFooter, $BranchID]);
  } else {
    $q = "INSERT INTO BranchSettings (BranchID, TaxRate, ServicePercent, Currency, PrintFooter)
          VALUES (?,?,?,?,?)";
    $pdo->prepare($q)->execute([$BranchID, $TaxRate, $ServicePercent, $Currency, $PrintFooter]);
  }
  json_out(['ok'=>true,'msg'=>'تم الحفظ بنجاح']);
}

/* ================= Initial Data ================= */
$user = current_user();
$defaultBranch = (int)($user['branch_id'] ?? 0);

// قائمة الفروع النشطة
$branches = $pdo->query("SELECT BranchID, BranchName FROM branches WHERE IsActive=1 ORDER BY BranchName")
                ->fetchAll(PDO::FETCH_ASSOC);
if (!$defaultBranch && !empty($branches)) {
  $defaultBranch = (int)$branches[0]['BranchID'];
}
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="utf-8">
  <title>⚙️ إعدادات الفروع</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- Bootstrap RTL + Icons -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.rtl.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <style>
    :root{
      --brand1:#0f2027; --brand2:#203a43; --brand3:#2c5364;
      --primary:#1abc9c; --primary-600:#16a085; --muted:#6b7280;
      --card:#ffffff; --bg:#f5f6f8; --border:rgba(0,0,0,.08);
    }
    body{background:var(--bg); font-family:"Tajawal", Tahoma, Arial, sans-serif; color:#1f2937}
    header{
      background:linear-gradient(135deg,var(--brand1),var(--brand2),var(--brand3));
      color:#fff; padding:14px 18px; display:flex; align-items:center; justify-content:space-between; gap:12px;
      box-shadow:0 6px 24px rgba(0,0,0,.25); margin-bottom:18px;
    }
    .user-info{display:flex;align-items:center;gap:10px}
    .user-info img{width:38px;height:38px;border-radius:50%;object-fit:cover;border:2px solid #fff}
    .panel{background:var(--card); border:1px solid var(--border); border-radius:14px; box-shadow:0 10px 24px rgba(0,0,0,.05);}
    .panel h6{margin:0;font-weight:800}
    .form-control, .form-select{border-radius:10px}
    .label-muted{color:var(--muted); font-size:.9rem}
    .btn-soft{background:#eef2ff; color:#1e40af; border:none}
    .receipt{background:#fff; border:1px dashed #d1d5db; border-radius:10px; padding:16px; font-family: ui-monospace, SFMono-Regular, Menlo, monospace}
    .receipt h5{font-weight:800; margin:0 0 8px}
    .receipt small{color:#6b7280}
    .footer-note{white-space:pre-wrap; line-height:1.6}
    .hint{color:var(--muted)}
    .badge-soft{background:#ecfdf5;color:#065f46;font-weight:700}
  </style>
</head>
<body>
<header class="container-fluid">
  <div class="user-info">
    <img src="uploads/<?= htmlspecialchars($_SESSION['avatar'] ?? 'default.png') ?>" alt="Avatar">
    <div>مرحبًا، <strong><?= htmlspecialchars($_SESSION['admin_username'] ?? 'مستخدم') ?></strong></div>
  </div>
  <div class="d-flex align-items-center gap-2">
    <a href="dashboard.php" class="btn btn-light btn-sm"><i class="fa-solid fa-house"></i> الرئيسية</a>
    <a href="logout.php" class="btn btn-outline-light btn-sm"><i class="fa-solid fa-arrow-right-from-bracket"></i> خروج</a>
  </div>
</header>

<div class="container">
  <!-- اختيار الفرع -->
  <div class="panel p-3 mb-3">
    <div class="row g-3 align-items-end">
      <div class="col-md-6">
        <label class="form-label">اختر الفرع</label>
        <select id="branchSel" class="form-select">
          <?php foreach ($branches as $b): ?>
            <option value="<?= (int)$b['BranchID'] ?>" <?= $b['BranchID']===$defaultBranch ? 'selected' : '' ?>>
              <?= htmlspecialchars($b['BranchName']) ?>
            </option>
          <?php endforeach; ?>
        </select>
        <div class="form-text">سيتم حفظ الإعدادات على هذا الفرع فقط.</div>
      </div>
      <div class="col-md-6 text-md-end">
        <span class="badge badge-soft"><i class="fa-solid fa-circle-info"></i> تلميح: أدخل القيم بالنِسب (%).</span>
      </div>
    </div>
  </div>

  <div class="row g-3">
    <!-- النموذج -->
    <div class="col-lg-7">
      <div class="panel p-3">
        <div class="d-flex justify-content-between align-items-center mb-2">
          <h6><i class="fa-solid fa-sliders"></i> إعدادات الفرع</h6>
          <button class="btn btn-outline-secondary btn-sm" id="btnDefaults">
            <i class="fa-solid fa-rotate"></i> استرجاع القيم الافتراضية
          </button>
        </div>

        <form id="settingsForm" class="row g-3" onsubmit="return false;">
          <input type="hidden" id="csrf" value="<?= htmlspecialchars($CSRF) ?>">

          <div class="col-md-6">
            <label class="form-label">نسبة الضريبة (٪)</label>
            <input type="number" step="0.01" min="0" max="100" class="form-control" id="taxRate" required>
            <div class="form-text label-muted">مثال: 14.00</div>
          </div>

          <div class="col-md-6">
            <label class="form-label">نسبة خدمة الصالة (٪)</label>
            <input type="number" step="0.01" min="0" max="100" class="form-control" id="servicePercent" required>
            <div class="form-text label-muted">مثال: 12.00 — اتركها 0 إن لم تُطبق.</div>
          </div>

          <div class="col-md-6">
            <label class="form-label">العملة</label>
            <select class="form-select" id="currency" required>
              <option value="EGP">EGP</option>
              <option value="SAR">SAR</option>
              <option value="AED">AED</option>
              <option value="USD">USD</option>
              <option value="EUR">EUR</option>
            </select>
            <div class="form-text label-muted">اضبط العملة الافتراضية للفرع.</div>
          </div>

          <div class="col-12">
            <label class="form-label">تذييل الطباعة (يظهر أسفل الفاتورة)</label>
            <textarea id="printFooter" rows="5" class="form-control" placeholder="مثال: شكراً لزيارتكم 🌟&#10;الأسعار شاملة الضريبة..."></textarea>
            <div class="form-text label-muted">يمكن استخدام سطر جديد. يُفضّل نص بسيط (بدون تنسيقات HTML معقدة).</div>
          </div>

          <div class="col-12 d-flex gap-2">
            <button id="btnSave" class="btn btn-success">
              <i class="fa-solid fa-floppy-disk"></i> حفظ الإعدادات
            </button>
            <button id="btnPreview" class="btn btn-soft">
              <i class="fa-solid fa-receipt"></i> معاينة الإيصال
            </button>
          </div>
        </form>

        <div id="alertBox" class="alert mt-3 d-none" role="alert"></div>
      </div>
    </div>

    <!-- المعاينة -->
    <div class="col-lg-5">
      <div class="panel p-3">
        <div class="d-flex justify-content-between align-items-center mb-2">
          <h6><i class="fa-solid fa-eye"></i> معاينة الإيصال</h6>
          <small class="hint">عرض تقريبي للطباعة</small>
        </div>
        <div class="receipt" id="preview">
          <h5 id="p_name">اسم الفرع</h5>
          <small id="p_meta">ضريبة: 0% • خدمة: 0% • عملة: EGP</small>
          <hr>
          <div class="footer-note" id="p_footer">—</div>
        </div>
      </div>
    </div>
  </div>
</div>

<!-- JS -->
<script src="https://code.jquery.com/jquery-3.7.0.min.js"></script>
<script>
  const csrf = document.getElementById('csrf').value;
  const $branchSel = $('#branchSel');
  const $tax = $('#taxRate');
  const $svc = $('#servicePercent');
  const $cur = $('#currency');
  const $footer = $('#printFooter');

  const $alert = $('#alertBox');
  function showAlert(type, msg){
    $alert.removeClass('d-none alert-success alert-danger alert-warning')
          .addClass('alert-' + type).text(msg);
    setTimeout(()=> $alert.addClass('d-none'), 3500);
  }

  function loadSettings(){
    const bid = $branchSel.val();
    if (!bid) return;
    $.getJSON('?action=get&branch_id=' + encodeURIComponent(bid), (resp)=>{
      if (!resp.ok) { showAlert('danger', resp.msg || 'تعذر جلب الإعدادات'); return; }
      const d = resp.data;
      $tax.val((+d.TaxRate).toFixed(2));
      $svc.val((+d.ServicePercent).toFixed(2));
      $cur.val(d.Currency || 'EGP');
      $footer.val(d.PrintFooter || '');
      updatePreview(d.BranchName, +d.TaxRate, +d.ServicePercent, d.Currency || 'EGP', d.PrintFooter || '');
    }).fail(()=> showAlert('danger','خطأ في الاتصال'));
  }

  function updatePreview(name, tax, svc, currency, footer){
    $('#p_name').text(name || 'اسم الفرع');
    $('#p_meta').text(`ضريبة: ${(+tax||0)}% • خدمة: ${(+svc||0)}% • عملة: ${currency||'EGP'}`);
    $('#p_footer').text(footer && footer.trim() ? footer : '—');
  }

  // أول تحميل
  $(function(){
    loadSettings();
  });

  // عند تغيير الفرع
  $branchSel.on('change', loadSettings);

  // زر الافتراضي
  $('#btnDefaults').on('click', function(){
    $tax.val('0.00');
    $svc.val('0.00');
    $cur.val('EGP');
    $footer.val('شكراً لزيارتكم 🌟\nالأسعار شاملة الضريبة.\nللاستفسار: 0000000000');
    updatePreview($('#branchSel option:selected').text(), 0, 0, 'EGP', $footer.val());
  });

  // معاينة
  $('#btnPreview').on('click', function(){
    updatePreview(
      $('#branchSel option:selected').text(),
      parseFloat($tax.val()||0),
      parseFloat($svc.val()||0),
      $cur.val(),
      $footer.val()
    );
  });

  // حفظ
  $('#btnSave').on('click', async function(){
    const bid = parseInt($branchSel.val()||'0', 10);
    if (!bid) { showAlert('warning','اختر الفرع أولًا'); return; }

    let tax = parseFloat($tax.val()||'0');
    let svc = parseFloat($svc.val()||'0');
    if (isNaN(tax) || tax<0 || tax>100) { showAlert('warning','قيمة الضريبة بين 0 و 100'); return; }
    if (isNaN(svc) || svc<0 || svc>100) { showAlert('warning','قيمة الخدمة بين 0 و 100'); return; }

    const payload = {
      action:'save',
      csrf: csrf,
      BranchID: bid,
      TaxRate: +tax.toFixed(2),
      ServicePercent: +svc.toFixed(2),
      Currency: $cur.val(),
      PrintFooter: $footer.val()
    };

    try{
      const res = await fetch('settings.php', {
        method:'POST',
        headers:{'Content-Type':'application/json'},
        body: JSON.stringify(payload)
      });
      const data = await res.json();
      if (!data.ok) { showAlert('danger', data.msg || 'فشل الحفظ'); return; }

      showAlert('success', 'تم الحفظ بنجاح');
      // حدّث المعاينة
      updatePreview($('#branchSel option:selected').text(), payload.TaxRate, payload.ServicePercent, payload.Currency, payload.PrintFooter);
    }catch(e){
      showAlert('danger','تعذر الاتصال بالخادم');
    }
  });
</script>
</body>
</html>
